<?php
// app/Modules/Driver/Services/DriverService.php

namespace App\Modules\Driver\Services;

use App\Modules\Driver\Models\Driver;
use App\Modules\Driver\Models\Vehicle;
use App\Modules\Driver\Models\DriverDocument;
use App\Modules\Driver\Models\DriverActivity;
use App\Modules\Driver\Models\Ride;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class DriverService
{

    /**
 * Get recent activities for a specific driver
 */
public function getRecentActivitiesForDriver($firebaseUid, $days = 7, $limit = 10)
{
    return DriverActivity::where('driver_firebase_uid', $firebaseUid)
                         ->where('created_at', '>=', now()->subDays($days))
                         ->orderBy('created_at', 'desc')
                         ->limit($limit)
                         ->get()
                         ->toArray();
}

/**
 * Get active rides for a specific driver
 */
public function getActiveRidesForDriver($firebaseUid)
{
    return Ride::where('driver_firebase_uid', $firebaseUid)
               ->whereIn('status', ['pending', 'accepted', 'in_progress'])
               ->get()
               ->toArray();
}

/**
 * Get admin dashboard data - FIXED VERSION
 */
public function getAdminDashboardData()
{
    try {
        $statistics = $this->getDriverStatistics();
        $systemAnalytics = $this->getSystemAnalytics();
        
        // Get recent drivers with error handling
        $recentDrivers = Driver::latest()->limit(10)->get()->map(function($driver) {
            return [
                'firebase_uid' => $driver->firebase_uid,
                'name' => $driver->name,
                'email' => $driver->email,
                'status' => $driver->status,
                'verification_status' => $driver->verification_status,
                'created_at' => $driver->created_at,
                'join_date' => $driver->join_date ?? $driver->created_at
            ];
        })->toArray();

        // Get recent activities with driver info
        $recentActivities = DriverActivity::with('driver')
                                         ->latest()
                                         ->limit(20)
                                         ->get()
                                         ->map(function($activity) {
                                             return [
                                                 'id' => $activity->id,
                                                 'title' => $activity->title,
                                                 'description' => $activity->description,
                                                 'activity_type' => $activity->activity_type,
                                                 'created_at' => $activity->created_at,
                                                 'driver_name' => $activity->driver->name ?? 'Unknown Driver',
                                                 'driver_firebase_uid' => $activity->driver_firebase_uid
                                             ];
                                         })
                                         ->toArray();

        // Get pending verifications
        $pendingVerifications = Driver::where('verification_status', Driver::VERIFICATION_PENDING)
                                    ->limit(10)
                                    ->get()
                                    ->map(function($driver) {
                                        return [
                                            'firebase_uid' => $driver->firebase_uid,
                                            'name' => $driver->name,
                                            'email' => $driver->email,
                                            'created_at' => $driver->created_at,
                                            'verification_status' => $driver->verification_status
                                        ];
                                    })
                                    ->toArray();

        // Get expiring documents with driver info
        $expiringDocuments = DriverDocument::whereBetween('expiry_date', [now(), now()->addDays(30)])
                                          ->with('driver')
                                          ->limit(10)
                                          ->get()
                                          ->map(function($document) {
                                              return [
                                                  'id' => $document->id,
                                                  'document_type' => $document->document_type,
                                                  'document_name' => $document->document_name,
                                                  'expiry_date' => $document->expiry_date,
                                                  'driver_name' => $document->driver->name ?? 'Unknown Driver',
                                                  'driver_firebase_uid' => $document->driver_firebase_uid
                                              ];
                                          })
                                          ->toArray();

        return [
            'statistics' => $statistics,
            'system_analytics' => $systemAnalytics,
            'recent_drivers' => $recentDrivers,
            'recent_activities' => $recentActivities,
            'pending_verifications' => $pendingVerifications,
            'expiring_documents' => $expiringDocuments
        ];
        
    } catch (\Exception $e) {
        Log::error('Error in getAdminDashboardData: ' . $e->getMessage());
        
        // Return minimal safe data structure if error occurs
        return [
            'statistics' => [
                'total_drivers' => 0,
                'active_drivers' => 0,
                'verified_drivers' => 0,
                'pending_verification' => 0
            ],
            'system_analytics' => [
                'driver_statistics' => [],
                'activity_statistics' => [],
                'document_statistics' => [],
                'vehicle_statistics' => [],
                'ride_statistics' => []
            ],
            'recent_drivers' => [],
            'recent_activities' => [],
            'pending_verifications' => [],
            'expiring_documents' => [],
            'error' => $e->getMessage()
        ];
    }
    
}
    /**
     * Get all drivers with filters
     */
    public function getAllDrivers(array $filters = [])
    {
        $query = Driver::query();

        // Apply filters
        if (!empty($filters['search'])) {
            $query->where(function ($q) use ($filters) {
                $q->where('name', 'like', '%' . $filters['search'] . '%')
                  ->orWhere('email', 'like', '%' . $filters['search'] . '%')
                  ->orWhere('phone', 'like', '%' . $filters['search'] . '%')
                  ->orWhere('license_number', 'like', '%' . $filters['search'] . '%');
            });
        }

        if (!empty($filters['status'])) {
            $query->where('status', $filters['status']);
        }

        if (!empty($filters['verification_status'])) {
            $query->where('verification_status', $filters['verification_status']);
        }

        if (!empty($filters['availability_status'])) {
            $query->where('availability_status', $filters['availability_status']);
        }

        $limit = $filters['limit'] ?? 50;
        
        return $query->paginate($limit);
    }

    /**
     * Get driver by Firebase UID
     */
    public function getDriverById($firebaseUid)
    {
        return Driver::where('firebase_uid', $firebaseUid)->first();
    }

    /**
     * Create new driver
     */
    public function createDriver(array $data)
    {
        DB::beginTransaction();
        try {
            $driver = Driver::create($data);
            
            // Create welcome activity
            DriverActivity::createActivity($driver->firebase_uid, DriverActivity::TYPE_PROFILE_UPDATE, [
                'title' => 'Welcome to the Platform',
                'description' => 'Your driver profile has been created'
            ]);

            DB::commit();
            return $driver;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating driver: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Update driver
     */
    public function updateDriver($firebaseUid, array $data)
    {
        $driver = $this->getDriverById($firebaseUid);
        
        if (!$driver) {
            return false;
        }

        $driver->update($data);
        
        // Create activity
        DriverActivity::createActivity($firebaseUid, DriverActivity::TYPE_PROFILE_UPDATE, [
            'title' => 'Profile Updated',
            'description' => 'Driver profile information has been updated'
        ]);

        return true;
    }

    /**
     * Update driver status
     */
    public function updateDriverStatus($firebaseUid, $status)
    {
        $driver = $this->getDriverById($firebaseUid);
        
        if (!$driver) {
            return false;
        }

        $driver->update(['status' => $status]);
        
        // Create activity
        DriverActivity::createActivity($firebaseUid, DriverActivity::TYPE_STATUS_CHANGE, [
            'title' => 'Status Changed',
            'description' => "Driver status changed to: " . ucfirst($status),
            'metadata' => ['new_status' => $status]
        ]);

        return true;
    }

    /**
     * Delete driver
     */
    public function deleteDriver($firebaseUid)
    {
        $driver = $this->getDriverById($firebaseUid);
        
        if (!$driver) {
            return false;
        }

        return $driver->delete();
    }

    /**
     * Get driver vehicles
     */
    public function getDriverVehicles($firebaseUid)
    {
        return Vehicle::where('driver_firebase_uid', $firebaseUid)->get();
    }

    /**
     * Get driver documents
     */
    public function getDriverDocuments($firebaseUid)
    {
        return DriverDocument::where('driver_firebase_uid', $firebaseUid)->get();
    }

    /**
     * Get driver rides
     */
    public function getDriverRides($firebaseUid, array $filters = [])
    {
        $query = Ride::where('driver_firebase_uid', $firebaseUid);

        if (!empty($filters['status'])) {
            $query->where('status', $filters['status']);
        }

        if (!empty($filters['date_from'])) {
            $query->where('created_at', '>=', $filters['date_from']);
        }

        if (!empty($filters['date_to'])) {
            $query->where('created_at', '<=', $filters['date_to']);
        }

        $limit = $filters['limit'] ?? 20;

        return $query->orderBy('created_at', 'desc')->paginate($limit);
    }

    /**
     * Get driver activities
     */
    public function getDriverActivities($firebaseUid, array $filters = [])
    {
        $query = DriverActivity::where('driver_firebase_uid', $firebaseUid);

        if (!empty($filters['activity_type'])) {
            $query->where('activity_type', $filters['activity_type']);
        }

        if (!empty($filters['unread_only'])) {
            $query->where('is_read', false);
        }

        $limit = $filters['limit'] ?? 30;

        return $query->orderBy('created_at', 'desc')->paginate($limit);
    }

    /**
     * Get driver statistics
     */
    public function getDriverStatistics()
    {
        return [
            'total_drivers' => Driver::count(),
            'active_drivers' => Driver::where('status', Driver::STATUS_ACTIVE)->count(),
            'verified_drivers' => Driver::where('verification_status', Driver::VERIFICATION_VERIFIED)->count(),
            'available_drivers' => Driver::where('availability_status', Driver::AVAILABILITY_AVAILABLE)
                                        ->where('status', Driver::STATUS_ACTIVE)
                                        ->count(),
            'pending_verification' => Driver::where('verification_status', Driver::VERIFICATION_PENDING)->count(),
            'recent_registrations' => Driver::where('created_at', '>=', now()->subDays(7))->count(),
            'total_rides' => Ride::count(),
            'completed_rides' => Ride::where('status', 'completed')->count(),
            'total_earnings' => Driver::sum('total_earnings'),
            'average_rating' => Driver::where('rating', '>', 0)->avg('rating')
        ];
    }

    /**
     * Get drivers near location
     */
    public function getDriversNearLocation($latitude, $longitude, $radius = 10)
    {
        return Driver::available()
                    ->nearby($latitude, $longitude, $radius)
                    ->get();
    }

    /**
     * Update driver location
     */
    public function updateDriverLocation($firebaseUid, $latitude, $longitude, $address = null)
    {
        $driver = $this->getDriverById($firebaseUid);
        
        if (!$driver) {
            return false;
        }

        $driver->updateLocation($latitude, $longitude, $address);

        // Create low priority activity
        DriverActivity::createActivity($firebaseUid, DriverActivity::TYPE_LOCATION_UPDATE, [
            'title' => 'Location Updated',
            'description' => 'Driver location has been updated',
            'location_latitude' => $latitude,
            'location_longitude' => $longitude,
            'location_address' => $address,
            'priority' => DriverActivity::PRIORITY_LOW
        ]);

        return true;
    }

    /**
     * Perform bulk action on drivers
     */
    public function performBulkAction($action, array $driverIds)
    {
        $processed = 0;
        $failed = 0;

        foreach ($driverIds as $driverId) {
            try {
                $driver = $this->getDriverById($driverId);
                
                if (!$driver) {
                    $failed++;
                    continue;
                }

                switch ($action) {
                    case 'activate':
                        $driver->update(['status' => Driver::STATUS_ACTIVE]);
                        break;
                    case 'deactivate':
                        $driver->update(['status' => Driver::STATUS_INACTIVE]);
                        break;
                    case 'suspend':
                        $driver->update(['status' => Driver::STATUS_SUSPENDED]);
                        break;
                    case 'verify':
                        $driver->update(['verification_status' => Driver::VERIFICATION_VERIFIED]);
                        break;
                    case 'delete':
                        $driver->delete();
                        break;
                    default:
                        $failed++;
                        continue 2;
                }

                $processed++;
            } catch (\Exception $e) {
                Log::error("Bulk action failed for driver {$driverId}: " . $e->getMessage());
                $failed++;
            }
        }

        return [
            'success' => true,
            'processed_count' => $processed,
            'failed_count' => $failed
        ];
    }

    /**
     * Get system analytics
     */
    public function getSystemAnalytics()
    {
        $last30Days = now()->subDays(30);
        $last7Days = now()->subDays(7);

        return [
            'driver_statistics' => $this->getDriverStatistics(),
            'activity_statistics' => [
                'total_activities' => DriverActivity::count(),
                'activities_today' => DriverActivity::whereDate('created_at', today())->count(),
                'activities_this_week' => DriverActivity::where('created_at', '>=', $last7Days)->count(),
                'activities_this_month' => DriverActivity::where('created_at', '>=', $last30Days)->count(),
                'unread_activities' => DriverActivity::where('is_read', false)->count()
            ],
            'document_statistics' => [
                'total_documents' => DriverDocument::count(),
                'pending_verification' => DriverDocument::where('verification_status', 'pending')->count(),
                'expired_documents' => DriverDocument::where('expiry_date', '<', now())->count(),
                'expiring_soon' => DriverDocument::whereBetween('expiry_date', [now(), now()->addDays(30)])->count()
            ],
            'vehicle_statistics' => [
                'total_vehicles' => Vehicle::count(),
                'active_vehicles' => Vehicle::where('status', Vehicle::STATUS_ACTIVE)->count(),
                'pending_verification' => Vehicle::where('verification_status', Vehicle::VERIFICATION_PENDING)->count()
            ],
            'ride_statistics' => [
                'total_rides' => Ride::count(),
                'rides_today' => Ride::whereDate('created_at', today())->count(),
                'rides_this_week' => Ride::where('created_at', '>=', $last7Days)->count(),
                'rides_this_month' => Ride::where('created_at', '>=', $last30Days)->count(),
                'completed_rides' => Ride::where('status', 'completed')->count(),
                'cancelled_rides' => Ride::where('status', 'cancelled')->count(),
                'total_revenue' => Ride::where('status', 'completed')->sum('total_amount')
            ]
        ];
    }


 

  

    /**
     * Get driver performance metrics
     */
    public function getDriverPerformanceMetrics($firebaseUid)
    {
        $driver = $this->getDriverById($firebaseUid);
        
        if (!$driver) {
            return null;
        }

        $last30Days = now()->subDays(30);

        return [
            'rating' => $driver->rating,
            'total_rides' => $driver->total_rides,
            'completed_rides' => $driver->completed_rides,
            'cancelled_rides' => $driver->cancelled_rides,
            'completion_rate' => $driver->completion_rate,
            'cancellation_rate' => $driver->cancellation_rate,
            'total_earnings' => $driver->total_earnings,
            'rides_last_30_days' => $driver->rides()->where('created_at', '>=', $last30Days)->count(),
            'earnings_last_30_days' => $driver->rides()
                                              ->where('created_at', '>=', $last30Days)
                                              ->where('status', 'completed')
                                              ->sum('driver_earnings'),
            'average_ride_duration' => $driver->rides()
                                              ->where('status', 'completed')
                                              ->avg('duration_minutes'),
            'average_ride_distance' => $driver->rides()
                                              ->where('status', 'completed')
                                              ->avg('distance_km')
        ];
    }

    /**
     * Export drivers data
     */
    public function exportDrivers(array $filters = [])
    {
        $query = Driver::query();

        if (!empty($filters['status'])) {
            $query->where('status', $filters['status']);
        }

        if (!empty($filters['verification_status'])) {
            $query->where('verification_status', $filters['verification_status']);
        }

        if (!empty($filters['created_from'])) {
            $query->where('created_at', '>=', $filters['created_from']);
        }

        if (!empty($filters['created_to'])) {
            $query->where('created_at', '<=', $filters['created_to']);
        }

        return $query->get()->map(function ($driver) {
            return [
                'firebase_uid' => $driver->firebase_uid,
                'name' => $driver->name,
                'email' => $driver->email,
                'phone' => $driver->phone,
                'status' => $driver->status,
                'verification_status' => $driver->verification_status,
                'availability_status' => $driver->availability_status,
                'rating' => $driver->rating,
                'total_rides' => $driver->total_rides,
                'total_earnings' => $driver->total_earnings,
                'join_date' => $driver->join_date,
                'last_active' => $driver->last_active
            ];
        })->toArray();
    }


    public function getDriversByVerificationStatus($status)
{
    return Driver::where('verification_status', $status)->get();
}

/**
 * Get drivers by status
 */
public function getDriversByStatus($status)
{
    return Driver::where('status', $status)->get();
}

/**
 * Get pending documents
 */
public function getPendingDocuments()
{
    return DriverDocument::where('verification_status', 'pending')->get();
}

/**
 * Get expired documents
 */
public function getExpiredDocuments()
{
    return DriverDocument::where('expiry_date', '<', now())->get();
}

/**
 * Get documents expiring soon (within 30 days)
 */
public function getDocumentsExpiringSoon()
{
    return DriverDocument::whereBetween('expiry_date', [now(), now()->addDays(30)])->get();
}

/**
 * Update driver verification status
 */
public function updateDriverVerificationStatus($firebaseUid, $status, $adminUid = null, $notes = null)
{
    $driver = $this->getDriverById($firebaseUid);
    
    if (!$driver) {
        return false;
    }

    $updateData = [
        'verification_status' => $status,
        'verified_at' => $status === 'verified' ? now() : null,
        'verified_by' => $adminUid
    ];

    if ($notes) {
        $updateData['verification_notes'] = $notes;
    }

    $driver->update($updateData);
    
    // Create activity
    DriverActivity::createActivity($firebaseUid, DriverActivity::TYPE_VERIFICATION_UPDATE, [
        'title' => 'Verification Status Updated',
        'description' => "Verification status changed to: " . ucfirst($status),
        'metadata' => [
            'new_status' => $status,
            'admin_uid' => $adminUid,
            'notes' => $notes
        ]
    ]);

    // Clear driver cache
    $this->clearDriverCache($firebaseUid);

    return true;
}

/**
 * Clear driver cache
 */
public function clearDriverCache($firebaseUid)
{
    $cacheKeys = [
        "driver_details_{$firebaseUid}",
        "driver_secondary_{$firebaseUid}",
        "driver_dynamic_{$firebaseUid}",
        "driver_recent_rides_{$firebaseUid}",
        "driver_all_rides_{$firebaseUid}",
        "driver_all_activities_{$firebaseUid}"
    ];

    foreach ($cacheKeys as $key) {
        cache()->forget($key);
    }

    // Clear general driver caches
    cache()->forget('driver_statistics');
    cache()->forget('total_drivers_count');
}

/**
 * Get total drivers count
 */
public function getTotalDriversCount()
{
    return Driver::count();
}

/**
 * Get driver profile completion percentage
 */
public function getDriverProfileCompletion($firebaseUid)
{
    $driver = $this->getDriverById($firebaseUid);
    
    if (!$driver) {
        return 0;
    }

    $requiredFields = [
        'name', 'email', 'phone', 'date_of_birth', 'gender', 
        'address', 'city', 'state', 'license_number', 'license_expiry'
    ];

    $completedFields = 0;
    foreach ($requiredFields as $field) {
        if (!empty($driver->$field)) {
            $completedFields++;
        }
    }

    return round(($completedFields / count($requiredFields)) * 100);
}

/**
 * Get driver ride statistics
 */
public function getDriverRideStatistics($firebaseUid)
{
    $driver = $this->getDriverById($firebaseUid);
    
    if (!$driver) {
        return null;
    }

    return [
        'total_rides' => $driver->total_rides ?? 0,
        'completed_rides' => $driver->completed_rides ?? 0,
        'cancelled_rides' => $driver->cancelled_rides ?? 0,
        'completion_rate' => $driver->completion_rate ?? 0,
        'total_earnings' => $driver->total_earnings ?? 0,
        'average_rating' => $driver->rating ?? 0
    ];
}

/**
 * Activate driver
 */
public function activateDriver($firebaseUid, $adminUid = null)
{
    $driver = $this->getDriverById($firebaseUid);
    
    if (!$driver) {
        return ['success' => false, 'message' => 'Driver not found'];
    }

    // Check profile completion
    $completion = $this->getDriverProfileCompletion($firebaseUid);
    
    if ($completion < 80) {
        return [
            'success' => false, 
            'message' => 'Driver profile must be at least 80% complete before activation',
            'completion_status' => $completion
        ];
    }

    // Update status
    $driver->update([
        'status' => Driver::STATUS_ACTIVE,
        'activated_at' => now(),
        'activated_by' => $adminUid
    ]);

    // Create activity
    DriverActivity::createActivity($firebaseUid, DriverActivity::TYPE_STATUS_CHANGE, [
        'title' => 'Account Activated',
        'description' => 'Your driver account has been activated',
        'metadata' => ['admin_uid' => $adminUid]
    ]);

    // Clear cache
    $this->clearDriverCache($firebaseUid);

    return ['success' => true, 'message' => 'Driver activated successfully'];
}

/**
 * Bulk import drivers
 */
public function bulkImportDrivers(array $driversData)
{
    $imported = 0;
    $failed = 0;

    DB::beginTransaction();
    try {
        foreach ($driversData as $driverData) {
            try {
                if (empty($driverData['firebase_uid']) || empty($driverData['email'])) {
                    $failed++;
                    continue;
                }

                // Check if driver already exists
                if (Driver::where('firebase_uid', $driverData['firebase_uid'])->exists()) {
                    $failed++;
                    continue;
                }

                Driver::create($driverData);
                $imported++;
            } catch (\Exception $e) {
                Log::error('Import driver failed: ' . $e->getMessage());
                $failed++;
            }
        }

        DB::commit();
        return [
            'success' => true,
            'imported_count' => $imported,
            'failed_count' => $failed
        ];
    } catch (\Exception $e) {
        DB::rollBack();
        Log::error('Bulk import failed: ' . $e->getMessage());
        return [
            'success' => false,
            'message' => 'Bulk import failed: ' . $e->getMessage()
        ];
    }
}

/**
 * Sync Firebase drivers
 */
public function syncFirebaseDrivers()
{
    // This would typically sync with Firebase
    // Implementation depends on your Firebase setup
    return [
        'success' => true,
        'synced_count' => 0,
        'message' => 'No new drivers to sync'
    ];
}
}